use criterion::{black_box, criterion_group, criterion_main, Criterion};
use hsv::chars::*;

fn generate_test_data(records: usize) -> String {
    let mut data = String::new();
    data.push(STX);

    for i in 0..records {
        if i > 0 {
            data.push(FS);
        }
        data.push_str(&format!(
            "id{US}{i}{RS}name{US}User {i}{RS}email{US}user{i}@example.com{RS}tags{US}tag1{GS}tag2{GS}tag3"
        ));
    }

    data.push(ETX);
    data
}

fn generate_nested_data(records: usize) -> String {
    let mut data = String::new();
    data.push(STX);

    for i in 0..records {
        if i > 0 {
            data.push(FS);
        }
        data.push_str(&format!(
            "user{US}{SO}id{US}{i}{RS}name{US}User {i}{RS}profile{US}{SO}bio{US}Hello world{RS}location{US}Earth{SI}{SI}"
        ));
    }

    data.push(ETX);
    data
}

fn bench_parse_simple(c: &mut Criterion) {
    let data_100 = generate_test_data(100);
    let data_1000 = generate_test_data(1000);
    let data_10000 = generate_test_data(10000);

    let mut group = c.benchmark_group("parse_simple");

    group.bench_function("100_records", |b| {
        b.iter(|| hsv::parse(black_box(&data_100)))
    });

    group.bench_function("1000_records", |b| {
        b.iter(|| hsv::parse(black_box(&data_1000)))
    });

    group.bench_function("10000_records", |b| {
        b.iter(|| hsv::parse(black_box(&data_10000)))
    });

    group.finish();
}

fn bench_parse_nested(c: &mut Criterion) {
    let data_100 = generate_nested_data(100);
    let data_1000 = generate_nested_data(1000);

    let mut group = c.benchmark_group("parse_nested");

    group.bench_function("100_records", |b| {
        b.iter(|| hsv::parse(black_box(&data_100)))
    });

    group.bench_function("1000_records", |b| {
        b.iter(|| hsv::parse(black_box(&data_1000)))
    });

    group.finish();
}

fn bench_sequential_vs_parallel(c: &mut Criterion) {
    let data = generate_test_data(10000);

    let mut group = c.benchmark_group("sequential_vs_parallel");

    group.bench_function("parallel_10000", |b| {
        b.iter(|| hsv::parse(black_box(&data)))
    });

    group.bench_function("sequential_10000", |b| {
        b.iter(|| hsv::parse_sequential(black_box(&data)))
    });

    group.finish();
}

criterion_group!(
    benches,
    bench_parse_simple,
    bench_parse_nested,
    bench_sequential_vs_parallel
);
criterion_main!(benches);
